﻿//----------------------------------------------------------
// Copyright (C) Microsoft Corporation. All rights reserved.
//----------------------------------------------------------
// TFS.Host.js
//
/// <reference path="~/Scripts/jquery-1.6.2-vsdoc.js" />
/// <reference path="~/Scripts/DevTime.js" />
/// <reference path="~/Scripts/MicrosoftAjax-4.0.0.0.debug.js"/>
/// <reference path="~/Scripts/TFS/TFS.debug.js" />
/// <reference path="~/Scripts/TFS/TFS.Core.debug.js" />
/// <reference path="~/Scripts/TFS/TFS.Diag.debug.js" />

TFS.module("TFS.Host", ["TFS.Resources.Common", "TFS.Core", "TFS.Core.Ajax", "TFS.Diag", "TFS.Core.Utils"], function () {
    var defaultTfsContext,
    notificationService,
    Diag = TFS.Diag,
    Core = TFS.Core,
    Utils = TFS.Core.Utils,
    CommonResources = TFS.Resources.Common,
    NavigationContextLevels,
    TeamFoundationHostType,
    handleError = TFS.handleError,
    history,
    imageTimeStamp,
    urlHelper,
    runningDocumentsTable,
    CommonActions,
    hostServiceManager;

    function setImageTimestamp() {
        imageTimeStamp = $.now();
        
    }

    function search(searchText, tfsContext) {
        if (searchText) {
            var context = tfsContext || TFS.Host.TfsContext.getDefault(),
                matchInteger = searchText.match(/^\s*#?(\d{1,9})\s*$/),
                workItemId = matchInteger ? parseInt(matchInteger[1]) : 0;

            if (workItemId > 0) {
                //search text is an integer and we are opening that work item
                TFS.using(["TFS.WorkItemTracking.Controls", "TFS.Host", "TFS.UI.Controls.Menus"], function () {
                    var MenuControls = TFS.UI.Controls.Menus;

                    MenuControls.menuManager.executeCommand(new Sys.CommandEventArgs("open-work-item", {
                        id: workItemId,
                        tfsContext: TFS.Host.TfsContext.getDefault()
                    }, null));
                });
            }
            else {
                if (context.navigation.project) {
                    window.location.href = context.getActionUrl("", "workItems") + urlHelper.getFragmentActionLink("search", { searchText: searchText });
                }
                else {
                    $("#select-project").addClass("validation-summary-errors");
                }
            }
        }
    }

    function Configuration(configData) {
        this._options = configData;
        /// <summary>Holds configuration information such as rootPath, theme, etc.</summary>
    }

    Configuration.prototype = {
        _options: {},

        getRootPath: function () {
            /// <summary>Gets the root path of the application</summary>
            /// <returns type="String" />
            return this._options.rootPath || "/";
        },

        getTheme: function () {
            /// <summary>Gets the current theme name</summary>
            /// <returns type="String" />
            return this._options.theme || "Default";
        },

        getResourcesPath: function () {
            /// <summary>Gets the current resource path where images, css files, etc. live</summary>
            /// <returns type="String" />
            return this._options.resourcesPath || (this.getRootPath() + "_content/");
        },

        getResourcesFile: function (file) {
            /// <summary>Converts the given the file into a resource relative path </summary>
            /// <param name="file" type="String">File to convert</param>
            /// <returns type="String" />
            return this.getResourcesPath() + encodeURIComponent(file);
        },

        getTfsStaticRootPath: function () {
            /// <summary>Returns URL path to TFS static files
            /// <returns type="String" />
            return this._options.staticRootTfs || "/tfs/_static/tfs/11/";
        },

        get3rdPartyStaticRootPath: function () {
            /// <summary>Returns URL path to 3rdParty static files
            /// <returns type="String" />
            return this._options.staticRoot3rdParty || "/tfs/_static/3rdParty/";
        },

        getThemedFile: function (file) {
            /// <summary>Converts the given the file into a theme relative path</summary>
            /// <param name="file" type="String">File to convert</param>
            /// <returns type="String" />
            return this.getTfsStaticRootPath() +
                    "App_Themes/" +
                    encodeURIComponent(this.getTheme()) + "/" +
                    encodeURIComponent(file);
        },

        getWebApiVersion: function () {
            /// <summary>Get Web Api version for TFS </summary>
            /// <returns type="String" />
            return this._options.webApiVersion || "1";
        },

        getMailSettings: function () {
            return this._options.mailSettings;
        }
    };


    function HostService() {
        /// <summary>A service available for host environment.  
        ///  A host can be tha browser, an IDE (e.g. Eclipse, Visual Studio) </summary>
    }

    HostService.extend(function () {
        return {
            getServiceName: function () {
                return this._serviceName || ("HostService." + this.getName());
            }
        };
    }());

    HostService.prototype = {
        getServiceName: function () {
            return this.constructor.getServiceName();
        }
    };

    function HostServiceManager() {
        this.services = {};
    }

    HostServiceManager.prototype = {
        getService: function (serviceType) {
            /// <summary>Get the specified service.  Example: hostServiceManager.getService(DocumentService) </summary>
            /// <param name="serviceType" type="Object">Service object type name</param>
            /// <returns type="Object" />
            var instance, serviceName;

            if (typeof serviceType.getServiceName === "undefined") {
                throw new Error("serviceType does not derive from HostService.");
            }

            serviceName = serviceType.getServiceName();
            instance = this.services[serviceName];

            if (!instance) {
                instance = new serviceType();
                this.services[serviceName] = instance;
            }
            return instance;
        }
    };

    // Singleton host service
    hostServiceManager = new HostServiceManager();

    function Document() {
        /// <summary>Represents a document to a host.
        ///  A host can be tha browser, an IDE (e.g. Eclipse, Visual Studio) </summary>
    }

    Document.prototype = {
        save: function (successCallback, errorCallback) { throw new Error("Not implemented"); },
        getMoniker: function () { throw new Error("Not implemented"); }
    };


    function DocumentService() {
        /// <summary>Service for host environment to interact with documents in Web Access
        ///  A host environment can be tha browser, an IDE (e.g. Eclipse, Visual Studio) </summary>
    }

    DocumentService.inherit(HostService, function () {
        var _activeDocument;

        return {

            addDeleteListener: function (callBack) {
                TFS.Host.notificationService.attachEvent("tfs-document-delete", callBack);
            },

            removeDeleteListener: function (callBack) {
                TFS.Host.notificationService.detachEvent("tfs-document-delete", callBack);
            },

            addBuildPropertyChangedListener: function (callBack) {
                TFS.Host.notificationService.attachEvent("tfs-document-build-property-changed", callBack);
            },

            removeBuildPropertyChangedListener: function (callBack) {
                TFS.Host.notificationService.detachEvent("tfs-document-build-property-changed", callBack);
            },

            addBuildStoppedListener: function (callBack) {
                TFS.Host.notificationService.attachEvent("tfs-document-build-stopped", callBack);
            },

            removeBuildStoppedListener: function (callBack) {
                TFS.Host.notificationService.detachEvent("tfs-document-build-stopped", callBack);
            },

            addModifiedChangedListener: function (callBack) {
                TFS.Host.notificationService.attachEvent("tfs-document-modified-change", callBack);
            },

            removeModifiedChangedListener: function (callBack) {
                TFS.Host.notificationService.detachEvent("tfs-document-modified-change", callBack);
            },

            isModified: function (args) {
                return TFS.Host.runningDocumentsTable.isModified(args);
            },

            save: function (successCallback, errorCallback) {
                var doc = this.getActiveDocument();

                if (doc) {
                    doc.save(successCallback, errorCallback);
                }
                else {
                    return TFS.Host.runningDocumentsTable.beginSave(successCallback, errorCallback);
                }
            },

            getActiveDocument: function () {
                return _activeDocument;
            },

            _setActiveDocument: function (activeDocument) {
                _activeDocument = activeDocument;
            }
        };
    }());

    NavigationContextLevels = {
        /// <summary>
        ///
        /// </summary>
        None: 0x00,

        /// <summary>
        /// Root level in Azure.
        /// </summary>
        Deployment: 0x01,

        /// <summary>
        /// Root level. Neither of {collection}, {project} and {team} tokens
        /// have information
        /// </summary>
        Application: 0x02,

        /// <summary>
        /// Flag to show {collection} token has information.
        /// </summary>
        Collection: 0x04,

        /// <summary>
        /// Flag to show {project} token has information.
        /// </summary>
        Project: 0x08,

        /// <summary>
        /// Flag to show {team} token has information.
        /// </summary>
        Team: 0x10,

        /// <summary>
        /// Sugar for all application levels.
        /// </summary>
        ApplicationAll: 0x0f,

        /// <summary>
        /// Sugar for all levels.
        /// </summary>
        All: 0x1f
    };

    TeamFoundationHostType =
    {
        Parent: -1,
        Unknown: 0,
        Deployment: 1,
        Application: 2,
        ProjectCollection: 4
    };

    function TfsContext(contextData) {
        this.contextData = contextData;
        this.navigation = contextData.navigation;
        this.configuration = new Configuration(contextData.configuration);
        this.currentUser = contextData.currentUser;
        this.currentIdentity = contextData.currentIdentity;
        this.currentTeam = contextData.currentTeam;
        this.currentUserHasTeamPermission = contextData.currentUserHasTeamPermission;
        this.standardAccessMode = contextData.standardAccessMode;
        this.isHosted = contextData.isHosted;
    }

    TfsContext.extend({
        _DEFAULT_CONTROLLER_NAME: "home",
        _DEFAULT_ACTION_NAME: "index",
        _CLIENTHOST: "clientHost",
        _VERSION: "__v",

        parseContext: function ($element) {

            // Getting the JSON string serialized by the server according to the current host
            var contextElement, json;

            contextElement = $element.find(".tfs-context");

            if (contextElement.length > 0) {
                json = contextElement.eq(0).html();

                if (json) {
                    return new TfsContext(Utils.parseMSJSON(json, false));
                }
            }

            return null;
        },

        getDefault: function () {
            if (!defaultTfsContext) {
                var context = this.parseContext($(document));

                Diag.assert(context !== null && typeof context !== "undefined", "Default context information is mising.");

                if (context) {
                    // Parsing the JSON and having it in the options
                    defaultTfsContext = context;
                } else {
                    // Create an empty TfsContext object if failed to get one from document
                    defaultTfsContext = new TfsContext({
                        navigation: null,
                        configuration: {},
                        currentUser: null,
                        currentIdentity: null,
                        currentTeam: null,
                        currentUserHasTeamPermission: false,
                        standardAccessMode: null,
                        isHosted: false
                    });
                }
            }

            return defaultTfsContext;
        },

        getContextOrDefault: function ($element) {
            var context;

            context = this.parseContext($element);

            if (!context) {
                return this.getDefault();
            }

            return context;
        },

        ControlExtensions: {
            getEnhancementOptions: function (element) {
                var options = this._base(element);
                return $.extend(options, { tfsContext: TfsContext.getContextOrDefault(element) });
            }
        }
    });

    TfsContext.prototype = {
        contextData: null,
        configuration: null,
        navigation: null,
        currentUser: null,
        currentIdentity: null,
        currentTeam: null,
        currentUserHasTeamPermission: false,
        standardAccessMode: null,
        isHosted: null,

        getPublicActionUrl: function (action, controller, routeData) {
            /// <summary>Constructs public action url using specified parameters and current TFS context</summary>
            /// <param name="action" type="String">Controller action</param>
            /// <param name="controller" type="String">Controller</param>
            /// <param name="routeData" type="Object">
            /// OPTIONAL: Information to use in constructing the url.  The object has the following structure:
            ///  {
            ///     team: <OPTIONAL: Defaulted to the current team in the navigation system if not provided.>
            ///     project: <OPTIONAL: Defaulted to the current project in the navigation system if not provided.>
            ///     serviceHost: <OPTIONAL: Defaulted to the current serviceHost (collection) in the navigation system if not provided.>
            ///     area: <OPTIONAL: Defaulted to the current area in the navigation system if not provided. Ex: "admin", or "api".>
            ///     includeVersion: <OPTIONAL: When set to true, the web client version will be included as part of the query string for the generated URL.
            ///                      This should only be used when generating URL's which will not show up in the address bar.
            ///                      NOTE: This is implicitly done when generating URL's in the API area.>
            ///  }
            /// </param>
            /// <returns type="String" />
            var actionUrl = this._constructActionUrl(action, controller, routeData);
            return this.navigation.publicAccessPoint.scheme + "://" + this.navigation.publicAccessPoint.authority + this.configuration.getRootPath() + actionUrl;
        },

        getActionUrl: function (action, controller, routeData) {
            /// <summary>Constructs action url using specified parameters and current TFS context</summary>
            /// <param name="action" type="String">Controller action</param>
            /// <param name="controller" type="String">Controller</param>
            /// <param name="routeData" type="Object">See routeData param on getPublicActionUrl.</param>
            /// <returns type="String" />
            var actionUrl = this._constructActionUrl(action, controller, routeData);
            return this.configuration.getRootPath() + actionUrl;
        },

        _constructActionUrl: function (action, controller, routeData) {
            var navigation = this.navigation,
                urlParts = [],
                serviceHost,
                project,
                team,
                area,
                routeParams,
                parameters,
                queryString;

            routeParams = $.extend({}, routeData);

            // --- BEGIN TFS Navigation Context
            serviceHost = routeParams.serviceHost;
            delete routeParams.serviceHost;

            if (typeof serviceHost === "undefined") {
                serviceHost = navigation.serviceHost;
            }

            if (serviceHost) {
                if (serviceHost.relVDir) {
                    urlParts.push(encodeURI(serviceHost.relVDir));
                }

                if (serviceHost.hostType === TeamFoundationHostType.ProjectCollection) {
                    project = routeParams.project;
                    delete routeParams.project;

                    if (typeof project === "undefined") {
                        project = navigation.project;
                    }

                    if (project) {
                        urlParts.push(encodeURIComponent(project));

                        team = routeParams.team;
                        delete routeParams.team;

                        if (typeof team === "undefined") {
                            team = navigation.team;
                        }

                        if (team) {
                            urlParts.push(encodeURIComponent(team));
                        }
                    }
                }
            }

            // --- END TFS Navigation Context

            // --- BEGIN AREA
            area = routeParams.area;

            if (typeof area === "undefined") {
                area = navigation.area;
            }

            delete routeParams.area;

            if (area) {
                urlParts.push(encodeURIComponent("_" + area));
            }

            // --- END AREA

            if (!controller) {
                controller = routeParams.controller;
            }

            delete routeParams.controller;

            if (!action) {
                action = routeParams.action;
            }

            delete routeParams.action;

            parameters = routeParams.parameters;
            delete routeParams.parameters;

            if (parameters) {
                // If the parameters are passed but no controller or no action then set defaults
                if (!controller) {
                    controller = TfsContext._DEFAULT_CONTROLLER_NAME;
                }

                if (!action) {
                    action = TfsContext._DEFAULT_ACTION_NAME;
                }
            }

            if (controller) {
                urlParts.push(encodeURIComponent("_" + controller));

                if (action) {
                    urlParts.push(encodeURIComponent(action));

                    if (parameters) {
                        if ($.isArray(parameters)) {
                            urlParts.push.apply(urlParts, $.map(parameters, encodeURIComponent));
                        }
                        else {
                            urlParts.push(encodeURIComponent(parameters));
                        }
                    }
                }
            }

            if (routeParams.includeVersion || area === "api") {
                routeParams[TfsContext._VERSION] = this.configuration.getWebApiVersion();
            }

            delete routeParams.includeVersion;

            // Add on client host query parameter if previously specified
            if (routeParams.area !== "api" && this.getClientHost()) {
                routeParams[TfsContext._CLIENTHOST] = this.getClientHost();
            }

            queryString = $.param(routeParams);

            return urlParts.join("/") + (queryString ? ("?" + queryString) : "");
        },

        getIdentityImageUrl: function (identityId, urlParams) {
            return this.getActionUrl("IdentityImage", "common", $.extend({ area: "api", project: "", id: identityId, t: imageTimeStamp }, urlParams));
        },

        isEmbedded: function () {
            /// <summary>Returns true if embedded in non browser host (e.g. TEE)
            /// </summary>
            /// <returns type="bool" />
            return (this.contextData.clientHost !== null);
        },

        getClientHost: function () {
            /// <summary>Get client host if any.  In normal browser
            /// scenario this is null.  If Web Access is running in
            /// context of a different host (e.g. TEE) this will have
            /// a value.</summary>
            /// <returns type="String" />
            return this.contextData.clientHost;
        }
    };

    function NotificationService() {
    }

    NotificationService.inheritBehavior(TFS.Core.EventListBehavior);

    $.extend(NotificationService.prototype, function () {
        return {
            fire: function (eventName, sender, eventArgs) {
                // Notifying all the subscribers
                return this._fireEvent(eventName, sender, eventArgs);
            }
        };
    }());

    // Creating the singleton instance of event subscription
    notificationService = new NotificationService();

    function createHistoryState(action, data) {
        if (data) {
            action = action || data._a || data.action;
            delete data.action;
            delete data._a;
        }

        if (action) {
            if (!data) {
                data = {};
            }

            data._a = action;
        }

        return data;
    }

    function History() {

    }

    History.prototype = {
        _events: null,
        _suppressNavigate: false,
        _initialized: false,
        _currentHashString: null,
        _ensureInitialized: function () {
            var that = this;

            if (!this._initialized) {
                this._initialized = true;

                this._events = new Sys.EventHandlerList();

                Sys.Application.add_navigate(function (sender, args) {
                    try {
                        if (that.checkCurrentState()) {
                            if (!that._suppressNavigate) {
                                that._onNavigate();
                            }
                        }
                    }
                    finally {
                        that._suppressNavigate = false;
                    }
                });
            }
        },
        _onNavigate: function () {
            var handler, historyState = this.getCurrentState();

            if (this._initialized) {
                if (historyState.action) {
                    handler = this._events.getHandler(historyState.action.toUpperCase());

                    if (handler) {
                        handler(this, historyState);
                    }
                }

                handler = this._events.getHandler("*");

                if (handler) {
                    handler(this, historyState);
                }
            }
        },
        getCurrentFragment: function () {
            return Sys.Application.get_stateString();
        },
        getCurrentState: function () {
            var state, action;
            this._currentHashString = this.getCurrentFragment();
            state = Sys.Application._deserializeState(this._currentHashString);
            action = state && state._a;

            if (action) {
                state.action = action;
                delete state._a;
            }

            return state;
        },
        checkCurrentState: function () {
            var stateString = this.getCurrentFragment();
            if (this._currentHashString != stateString) {
                this._currentHashString = stateString;
                return true;
            }

            return false;
        },
        replaceHistoryPoint: function (action, data, windowTitle, suppressNavigate) {
            var state = {};
            this._suppressNavigate = suppressNavigate === true;

            $.each(createHistoryState(action, data), function (name, value) {
                if (value != null) {
                    state[name] = value;
                }
            });

            window.location.replace("#" + Sys.Application._serializeState(state));
        },
        addHistoryPoint: function (action, data, windowTitle, suppressNavigate) {
            this._suppressNavigate = suppressNavigate === true;

            Sys.Application.addHistoryPoint(createHistoryState(action, data), windowTitle);
        },
        attachNavigate: function (action, handler, checkCurrentState) {
            var currentState;
            this._ensureInitialized();
            if ($.isFunction(action)) {
                this._events.addHandler("*", action);
            }
            else if (handler && action) {
                this._events.addHandler(action.toString().toUpperCase(), handler);

                if (checkCurrentState) {
                    currentState = this.getCurrentState();
                    if (currentState.action && action.toString().toUpperCase() === currentState.action.toUpperCase()) {
                        handler(this, currentState);
                    }
                }
            }
        },
        detachNavigate: function (action, handler) {
            if (this._initialized) {
                if ($.isFunction(action)) {
                    this._events.removeHandler("*", action);
                }
                else if (handler && action) {
                    this._events.removeHandler(action.toString().toUpperCase(), handler);
                }
            }
        }
    };

    history = new History();

    function UrlHelper() {
        this._urlTranslators = [];
    }

    UrlHelper.extend({
        //////////////////////////////////////////////////////////////////////////////////////////////////////////
        // IMPORTANT!!! The safe and unsafe URI scheme lists need to be in kept EXACTLY IN SYNC with the lists in
        // alm\tfs_core\Framework\Common\UriUtility\UriUtility.cs
        //////////////////////////////////////////////////////////////////////////////////////////////////////////

        // TODO - Should store the safe scheme list on the server, provide admin functionality for it and have
        // VS and Web Access use the same list.
        SAFE_URI_SCHEME_LIST: [
            "http", "https", "ftp", "gopher", "mailto", "news", "telnet", "wais",
            "vstfs", "tfs", "alm", "mtm", "mtms", "mfbclient", "mfbclients",
            "x-mvwit"
        ]
    });

    UrlHelper.prototype = {
        _urlTranslators: null,
        getFragmentActionLink: function (action, data) {
            /// <summary>Creates a fragment url to be used in flight navigation.</summary>
            /// <param name="action" type="String">The action name</param>
            /// <param name="data" type="Object">Action parameters</param>
            /// <returns type="String">fragment URL in the form of #_a=[action]&routevalue1=routevalue2...</returns>
            return "#" + Sys.Application._serializeState(createHistoryState(action, data));
        },

        isSafeProtocol: function (url) {
            /// <summar>Check if specified URL is safe - i.e. part of an approved list of URL schemes.</summary>
            /// <param name="url" type="String">Url to check</param>
            /// <returns type="boolean">
            var safeSchemeList = UrlHelper.SAFE_URI_SCHEME_LIST;
            var i, trimUrl, indexOfSchemeDelimiter, scheme;

            Diag.assert(typeof url === "string");
            trimUrl = url.trim();
            indexOfSchemeDelimiter = trimUrl.indexOf(":");

            if (indexOfSchemeDelimiter >= 0) {
                scheme = trimUrl.substr(0, indexOfSchemeDelimiter);

                for (i = 0; i < safeSchemeList.length; i++) {
                    if (String.ignoreCaseComparer(scheme, safeSchemeList[i]) == 0) {
                        return true;
                    }
                }
            }

            return false;
        },

        registerUrlTranslator: function (translatorFunction, order) {
            /// <summary>Registers a URL translator function.</summary>
            /// <param name="translatorFunction" type="Function">The translator function of the form function(url, options, successCallback, errorCallback, nextTranslator){}</param>
            /// <param name="order" type="Number">The order of the translator function.</param>

            this._urlTranslators.push({
                translator: translatorFunction,
                order: order || 100
            });

            this._urlTranslators.sort(function (ta, tb) {
                return ta.order - tb.order;
            });
        },
        beginTranslateUrl: function (url, options, callback, errorCallback) {
            var that = this,
                translatorIndex = 0,
                translators;

            function next() {
                var translatorEntry = that._urlTranslators[translatorIndex++];

                if (translatorEntry) {
                    translatorEntry.translator.call(that, url, options, succeeded, failed, next);
                } else {
                    succeeded(url);
                }
            }

            function failed(error) {
                handleError(error, errorCallback, that);
            }

            function succeeded(url) {
                if ($.isFunction(callback)) {
                    callback.call(that, url);
                }
            }

            if (url) {
                if (this.isSafeProtocol(url)) {
                    next();
                } else {
                    return succeeded(null);
                }
            } else {
                succeeded(url);
            }
        }
    };

    urlHelper = new UrlHelper();

    function ActionManager() {
        /// <summary>Manage actions and the workers that are invoked when those actions are performed.
        /// Action workers register to handle specific actions. They take whatever action they desire
        /// and usually call the "next" handler in the chain (see the Chain of Responsibility pattern).
        /// </summary>
    }

    ActionManager.extend(function () {
        var actionWorkers = {};

        return {
            registerActionWorker: function (action, actionWorker, order) {
                /// <summary>Register a handler for an action. The handler participates in the Chain of Responsibility pattern.</summary>
                /// <param name="action" type="String">The action to register</param>
                /// <param name="actionWorker" type="Function(actionArgs, next)">The handler to invoke for the given action when the <c>performAction</c> 
                ///     operation is called for the registered action.
                ///     The function is passed the action arguments for <c>next</c> which it should call with the <c>actionsArgs</c> UNLESS
                ///     it explicitly wants to be the end of the chain.
                ///     e.g. 
                ///     registerActionWorker('some.action', function (actionArgs, next) {
                ///         if (iCanHandle(actionArgs)) {
                ///             return doProcessing(actionArgs);
                ///         }
                ///         else {
                ///             return next(actionArgs);
                ///         }
                ///     }, 50);
                ///     
                /// if ActionWorker functions are asynchronous they can still participate in the chain
                /// 
                ///     registerActionWorker('some.async.action', function (actionArgs, next) {
                ///         beginDoSomeStuff(function (result) {
                ///             if (that.imDone(results)) {
                ///                 actionArgs.onSuccess.call(this, results);
                ///             }
                ///             else {
                ///                 next(actionArgs);
                ///             }
                ///         });
                ///     }, 50);
                /// </param>
                /// <param name="order" type="Number" optional="true">The order of the action (default:100). Action workers are executed in increasing order.</param>

                Diag.assertParamIsStringNotEmpty(action, "action");
                Diag.assertParamIsFunction(actionWorker, "actionWorker");
                if (order !== undefined) {
                    Diag.assertParamIsNumber(order, "order");
                }

                var workers = actionWorkers[action];

                if (!workers) {
                    actionWorkers[action] = workers = [];
                }

                workers.push({
                    worker: actionWorker,
                    order: order || 100
                });

                workers.sort(function (wa, wb) {
                    return wa.order - wb.order;
                });
            },

            performAction: function (action, actionArgs) {
                /// <summary>Invoke the registered action workers for the an action</summary>
                /// <param name="action" type="String">The action identifier</param>
                /// <param name="actionArgs" type="Object" optional="true">An object passed to the registered action workers.</param>

                Diag.assertParamIsStringNotEmpty(action, "action");

                var workers = actionWorkers[action],
                    index = 0;

                function next(actionArgs) {
                    var entry = workers && workers[index++];

                    if (entry) {
                        return entry.worker.call(this, actionArgs, next);
                    }
                }

                return next(actionArgs);
            }
        };
    }());

    CommonActions = {
        ACTION_WINDOW_OPEN: "window-open",
        ACTION_WINDOW_NAVIGATE: "window-navigate",
        ACTION_WINDOW_RELOAD: "window-reload",
        ACTION_WINDOW_UNLOAD: "window-unload"
    };

    function RunningDocumentsTableEntry(moniker, document) {
        this.document = document;
        this.moniker = moniker;
    }

    /// <summary>
    ///   Running Document Table (RDT) keeps track of open documents\items
    ///   in Web Access.  The document may actually be a document
    ///   manager.  This is used for cases where the state of
    ///   documents can affect behavior, e.g. prompt user if they leave
    ///   page when there are modified documents.
    ///
    ///   There is only one Running Documents Table accessible via
    ///   TFS.Host.runningDocumentsTable
    ///
    ///   The document added to the RDT must have the isDirty function
    ///
    /// </summary>
    function RunningDocumentsTable() {
        this.rdt = [];  // running document table
    }

    RunningDocumentsTable.prototype = {
        rdt: null,
        add: function (moniker, document) {
            /// <summary>
            ///   Add specified document to the running document table
            ///   The document must have a method named isDirty that returns bool
            /// </summary>
            /// <param name="moniker" type="String">Name for this document type</param>
            /// <param name="document" type="object">Object that will be called to determine state (e.g. dirty//modified)</param>
            /// <returns type="object">A handle to the entry in the running document table. The handle can be used to remove the entry</returns>
            ///
            var entry = new RunningDocumentsTableEntry(moniker, document);
            TFS.Diag.assertIsFunction(document["isDirty"], "RunningDocumentsTable.add() parameter document must have isDirty function");
            this.rdt.push(entry);
            return entry;
        },

        remove: function (entry) {
            /// <summary>
            ///   Remove an entry from the running document table
            /// </summary>
            /// <param name="entry" type="object">The handle to the entry that will be removed. The handle is returned from the <c>add</c> function</param>
            this.rdt = this.rdt.subtract([entry]);
        },

        isModified: function (moniker) {
            /// <summary>
            ///   Check if the specified document is modified.  If specified moniker is null or undefined
            ///   will return true if any currently opened documents are modified
            /// </summary>
            /// <param name="moniker" type="String">Name for this document type</param>
            /// <returns type="bool">True if the specified moniker\document is modified, false otherwise.
            ///   Null or undefined moniker will return true if any opened documents are modified</returns>
            if (moniker) {
                return this.rdt[moniker].isDirty();
            }

            return this._isAnyModified();
        },

        _isAnyModified: function () {
            var i, l, entry;
            for (i = 0, l = this.rdt.length; i < l; i++) {
                entry = this.rdt[i];
                if (entry.document.isDirty()) {
                    return true;
                }
            }
            return false;
        },

        beginSave: function (callback, errorCallback) {
            var that = this,
                dirtyDocs = [],
                index = 0;

            $.each(this.rdt, function (index, entry) {
                if (entry.document.isDirty() && $.isFunction(entry.document.beginSave)) {
                    dirtyDocs.push(entry.document);
                }
            });

            function next() {
                var document = dirtyDocs[index++];

                if (document) {
                    document.beginSave(function () {
                        Core.delay(that, 0, next);
                    }, failed);
                } else {
                    if ($.isFunction(callback)) {
                        callback.call(that);
                    }
                }
            }

            function failed(error) {
                handleError(error, errorCallback, that);
            }

            next();
        },

        getUnsavedItemsMessage: function () {
            var i, l, entry, titles, messages = [];

            for (i = 0, l = this.rdt.length; i < l; i++) {
                entry = this.rdt[i];
                if (entry.document.isDirty() && $.isFunction(entry.document.getDirtyDocumentTitles)) {
                    titles = entry.document.getDirtyDocumentTitles(6);
                    $.each(titles, function (titleIndex, title) {
                        if (messages.length == 5) {
                            messages.push(CommonResources.UnsavedChangesMore);
                            return false;
                        }
                        if (title.length > 55) {
                            title = title.substr(0, 52) + "...";
                        }
                        messages.push(title);
                    });
                    if (messages.length > 5) {
                        break;
                    }
                }
            }

            if (messages.length == 0) {
                return CommonResources.UnsavedChanges;
            }
            else {
                return CommonResources.UnsavedChangesWithNames + "\n" + messages.join("\n");
            }
        }
    };

    // Create the singleton running documents table
    runningDocumentsTable = new RunningDocumentsTable();

    return {
        search: search,
        NavigationContextLevels: NavigationContextLevels,
        TeamFoundationHostType: TeamFoundationHostType,
        TfsContext: TfsContext,
        setImageTimestamp: setImageTimestamp,
        history: history,
        urlHelper: urlHelper,
        notificationService: notificationService,
        ActionManager: ActionManager,
        CommonActions: CommonActions,
        runningDocumentsTable: runningDocumentsTable,
        DocumentService: DocumentService,
        Document: Document,
        hostServiceManager : hostServiceManager
    };
});

// SIG // Begin signature block
// SIG // MIIamQYJKoZIhvcNAQcCoIIaijCCGoYCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFIECpdBbCfT+
// SIG // RVmOGurl4+BGnHMboIIVeTCCBLowggOioAMCAQICCmEC
// SIG // jkIAAAAAAB8wDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OFoXDTEz
// SIG // MDQwOTIyMjU1OFowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpGNTI4LTM3NzctOEE3NjElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAJbsjkdN
// SIG // VMJclYDXTgs9v5dDw0vjYGcRLwFNDNjRRi8QQN4LpFBS
// SIG // EogLQ3otP+5IbmbHkeYDym7sealqI5vNYp7NaqQ/56ND
// SIG // /2JHobS6RPrfQMGFVH7ooKcsQyObUh8yNfT+mlafjWN3
// SIG // ezCeCjOFchvKSsjMJc3bXREux7CM8Y9DSEcFtXogC+Xz
// SIG // 78G69LPYzTiP+yGqPQpthRfQyueGA8Azg7UlxMxanMTD
// SIG // 2mIlTVMlFGGP+xvg7PdHxoBF5jVTIzZ3yrDdmCs5wHU1
// SIG // D92BTCE9djDFsrBlcylIJ9jC0rCER7t4utV0A97XSxn3
// SIG // U9542ob3YYgmM7RHxqBUiBUrLHUCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBQv6EbIaNNuT7Ig0N6JTvFH7kjB8jAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBz/30unc2NiCt8feNeFXHp
// SIG // aGLwCLZDVsRcSi1o2PlIEZHzEZyF7BLUVKB1qTihWX91
// SIG // 7sb1NNhUpOLQzHyXq5N1MJcHHQRTLDZ/f/FAHgybgOIS
// SIG // CiA6McAHdWfg+jSc7Ij7VxzlWGIgkEUvXUWpyI6zfHJt
// SIG // ECfFS9hvoqgSs201I2f6LNslLbldsR4F50MoPpwFdnfx
// SIG // Jd4FRxlt3kmFodpKSwhGITWodTZMt7MIqt+3K9m+Kmr9
// SIG // 3zUXzD8Mx90Gz06UJGMgCy4krl9DRBJ6XN0326RFs5E6
// SIG // Eld940fGZtPPnEZW9EwHseAMqtX21Tyi4LXU+Bx+BFUQ
// SIG // axj0kc1Rp5VlMIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSMMIIEiAIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIGuMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBSBOD2KgllKrMzqumqJtZke
// SIG // dB4dIDBOBgorBgEEAYI3AgEMMUAwPqAkgCIAVABGAFMA
// SIG // LgBIAG8AcwB0AC4AZABlAGIAdQBnAC4AagBzoRaAFGh0
// SIG // dHA6Ly9taWNyb3NvZnQuY29tMA0GCSqGSIb3DQEBAQUA
// SIG // BIIBAJqGAQNLWrOcO0Cr8dMHImbTMfdR4b5YT3nlfPSn
// SIG // Teq5etEc7lKctjhKh8wZq+zmTqXdaWp4hYmdcAvGHgqq
// SIG // fM/KgbwlAEN2GI+kGnrglayicMAUZy801ivEHG4Y4d6u
// SIG // IrjxeF0Z0tj/nvWXiSLam/ouYIPUlZZiK/m40G5IMmvy
// SIG // yS9DqphgRkJY24lVtA9FS2FB7ZEOLf5nDtMdw6jVAltD
// SIG // sZyGuW1PwrCxT8fn1MbmfQPZQUjGlU0IMhZbkPvRSDPS
// SIG // Y+aVDgIYOJW4dqWDda2BwKKtMIEhmf6Akm6eCjFOwQJU
// SIG // 0N+MfIEiyJxE/TLIahvc0RQh7qPOP9O45MiLF0+hggIf
// SIG // MIICGwYJKoZIhvcNAQkGMYICDDCCAggCAQEwgYUwdzEL
// SIG // MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
// SIG // EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9z
// SIG // b2Z0IFRpbWUtU3RhbXAgUENBAgphAo5CAAAAAAAfMAkG
// SIG // BSsOAwIaBQCgXTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcN
// SIG // AQcBMBwGCSqGSIb3DQEJBTEPFw0xMzAzMTUwNjMzNTVa
// SIG // MCMGCSqGSIb3DQEJBDEWBBREI+p4fG4VLHvg2HHmEWnk
// SIG // ZVwvUDANBgkqhkiG9w0BAQUFAASCAQCLH2cY5wb2re/A
// SIG // ZIJuUZE8lzc098nbXOaRbdF6CmU/rzeU7SqYCZVrRerU
// SIG // AX0AEqCQwkSpvKkYfgeBC7ron7d5Qz7ZfF+EpL7e2EQy
// SIG // 7f/1TfTcbQurc3hIfIxM98KNrbDUrBdrSlLiik39Kmto
// SIG // r3QDARM7UD6uSE+I/3tGuOb7RkiOuFYl1YzhYY1BlZIP
// SIG // CgtMAVykM0be5jriPM3+pFpSAf9gHbufndMVH2K8VnTO
// SIG // idInVteLHOAsMAtZ01EFxc0RU6K9Wkk+vTM6/plb0pBz
// SIG // tJ+8ZIh3G4olXPtQ1dSlbunW6wX41wt+C+h6nhmuZm7q
// SIG // mT1HG39vXgiqtj3ctFlq
// SIG // End signature block
